// Auto-Apply Logic for Job Seekers

let autoApplyActive = false;
let autoApplyInterval = null;
let applicationStats = {
    found: 0,
    applied: 0,
    skipped: 0,
    dailyApplications: 0
};

// LinkedIn job scanning and auto-apply
async function startAutoApplyProcess(preferences) {
    autoApplyActive = true;
    applicationStats = { found: 0, applied: 0, skipped: 0, dailyApplications: 0 };
    
    // Check daily limit
    const data = await chrome.storage.local.get(['dailyApplications', 'lastApplicationDate']);
    const today = new Date().toDateString();
    
    if (data.lastApplicationDate !== today) {
        // Reset daily counter
        await chrome.storage.local.set({
            dailyApplications: 0,
            lastApplicationDate: today
        });
        applicationStats.dailyApplications = 0;
    } else {
        applicationStats.dailyApplications = data.dailyApplications || 0;
    }
    
    // Start scanning for jobs
    autoApplyInterval = setInterval(async () => {
        if (!autoApplyActive) {
            clearInterval(autoApplyInterval);
            return;
        }
        
        // Check if we're on LinkedIn
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        const currentTab = tabs[0];
        
        if (currentTab && currentTab.url && currentTab.url.includes('linkedin.com')) {
            // Inject content script to scan for jobs
            chrome.tabs.sendMessage(currentTab.id, {
                action: 'scanForJobs',
                preferences: preferences
            });
        }
    }, 30000); // Check every 30 seconds
    
    // Initial scan
    scanLinkedInJobs(preferences);
}

async function scanLinkedInJobs(preferences) {
    try {
        // Get active tab
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        const currentTab = tabs[0];
        
        if (!currentTab || !currentTab.url) return;
        
        // Build search URL based on preferences
        const searchParams = new URLSearchParams();
        searchParams.set('keywords', preferences.jobTitles.join(' OR '));
        searchParams.set('location', preferences.locations[0] || 'India');
        searchParams.set('f_AL', 'true'); // Easy Apply only
        
        // Experience level mapping
        const expLevelMap = {
            'entry': '1,2',
            'mid': '3,4',
            'senior': '5,6',
            'any': '1,2,3,4,5,6'
        };
        searchParams.set('f_E', expLevelMap[preferences.experienceLevel] || '1,2,3,4,5,6');
        
        const searchUrl = `https://www.linkedin.com/jobs/search/?${searchParams.toString()}`;
        
        // Navigate to search page if not already there
        if (!currentTab.url.includes('/jobs/search/')) {
            await chrome.tabs.update(currentTab.id, { url: searchUrl });
            
            // Wait for page to load
            setTimeout(() => {
                chrome.tabs.sendMessage(currentTab.id, {
                    action: 'scanForJobs',
                    preferences: preferences
                });
            }, 3000);
        } else {
            // Already on jobs page, scan immediately
            chrome.tabs.sendMessage(currentTab.id, {
                action: 'scanForJobs',
                preferences: preferences
            });
        }
    } catch (error) {
        console.error('Error scanning LinkedIn jobs:', error);
    }
}

function stopAutoApplyProcess() {
    autoApplyActive = false;
    if (autoApplyInterval) {
        clearInterval(autoApplyInterval);
        autoApplyInterval = null;
    }
}

// Handle job application
async function applyToJob(jobData, preferences) {
    try {
        // Check daily limit
        if (applicationStats.dailyApplications >= 50) {
            console.log('Daily application limit reached');
            return { success: false, reason: 'Daily limit reached' };
        }
        
        // Get user's resume
        const userData = await chrome.storage.local.get(['userResume', 'userEmail']);
        if (!userData.userResume) {
            return { success: false, reason: 'No resume uploaded' };
        }
        
        // Calculate match score
        const matchScore = await calculateJobMatch(userData.userResume, jobData.description);
        
        if (matchScore < preferences.minScore) {
            applicationStats.skipped++;
            return { success: false, reason: 'Score too low', score: matchScore };
        }
        
        // Send application via content script
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        const response = await chrome.tabs.sendMessage(tabs[0].id, {
            action: 'applyToJob',
            jobId: jobData.id,
            jobData: jobData
        });
        
        if (response && response.success) {
            applicationStats.applied++;
            applicationStats.dailyApplications++;
            
            // Save application to history
            await saveApplication(jobData, matchScore);
            
            // Update daily counter
            await chrome.storage.local.set({
                dailyApplications: applicationStats.dailyApplications
            });
            
            // Notify popup
            chrome.runtime.sendMessage({
                action: 'updateAutoApplyStats',
                stats: applicationStats,
                recentApplication: {
                    company: jobData.company,
                    position: jobData.title
                }
            });
            
            return { success: true, score: matchScore };
        }
        
        return { success: false, reason: 'Application failed' };
    } catch (error) {
        console.error('Error applying to job:', error);
        return { success: false, reason: error.message };
    }
}

// Calculate job match score
async function calculateJobMatch(resumeText, jobDescription) {
    try {
        // Use the existing scoring logic from background.js
        const response = await fetch(`${CONFIG.BACKEND_URL}/api/score`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${authToken}`
            },
            body: JSON.stringify({
                resumeText: resumeText,
                jobDescription: jobDescription
            })
        });
        
        if (response.ok) {
            const result = await response.json();
            return result.overallScore || 0;
        }
        
        // Fallback to simple keyword matching
        return calculateSimpleMatch(resumeText, jobDescription);
    } catch (error) {
        console.error('Error calculating match:', error);
        return calculateSimpleMatch(resumeText, jobDescription);
    }
}

// Simple keyword matching fallback
function calculateSimpleMatch(resumeText, jobDescription) {
    const resumeLower = resumeText.toLowerCase();
    const jobLower = jobDescription.toLowerCase();
    
    // Extract keywords from job description
    const keywords = jobLower.match(/\b\w+\b/g) || [];
    const importantKeywords = keywords.filter(word => word.length > 4);
    
    // Count matches
    let matches = 0;
    importantKeywords.forEach(keyword => {
        if (resumeLower.includes(keyword)) {
            matches++;
        }
    });
    
    // Calculate percentage
    const score = Math.min(100, Math.round((matches / importantKeywords.length) * 100));
    return score || 50; // Default to 50% if can't calculate
}

// Save application to history
async function saveApplication(jobData, score) {
    const data = await chrome.storage.local.get(['applicationHistory']);
    const history = data.applicationHistory || [];
    
    history.unshift({
        id: jobData.id,
        company: jobData.company,
        position: jobData.title,
        location: jobData.location,
        score: score,
        appliedAt: new Date().toISOString(),
        status: 'applied',
        url: jobData.url
    });
    
    // Keep only last 100 applications
    if (history.length > 100) {
        history.pop();
    }
    
    await chrome.storage.local.set({ applicationHistory: history });
}

// Export for use in background.js
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        startAutoApplyProcess,
        stopAutoApplyProcess,
        applyToJob,
        calculateJobMatch
    };
}