// LinkedIn Auto-Apply Content Script

// Listen for messages from extension
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'scanForJobs') {
        scanJobListings(request.preferences);
        sendResponse({ success: true });
    } else if (request.action === 'applyToJob') {
        applyToSpecificJob(request.jobId, request.jobData).then(sendResponse);
        return true; // Keep channel open for async response
    }
});

// Scan job listings on current page
async function scanJobListings(preferences) {
    console.log('Scanning for jobs with preferences:', preferences);
    
    // Find all job cards on the page
    const jobCards = document.querySelectorAll('.jobs-search-results__list-item, .job-card-container');
    
    const jobs = [];
    
    jobCards.forEach(card => {
        try {
            // Extract job data
            const titleElement = card.querySelector('.job-card-list__title, .job-card-container__link');
            const companyElement = card.querySelector('.job-card-container__company-name, .job-card-container__primary-description');
            const locationElement = card.querySelector('.job-card-container__metadata-item, .job-card-list__location');
            const easyApplyElement = card.querySelector('.job-card-container__apply-method, [aria-label*="Easy Apply"]');
            
            if (titleElement && easyApplyElement) {
                const jobData = {
                    id: card.dataset.jobId || card.id || generateJobId(),
                    title: titleElement.textContent.trim(),
                    company: companyElement?.textContent.trim() || 'Unknown Company',
                    location: locationElement?.textContent.trim() || 'Unknown Location',
                    url: titleElement.href || window.location.href,
                    hasEasyApply: true
                };
                
                // Check if job matches preferences
                if (matchesPreferences(jobData, preferences)) {
                    jobs.push(jobData);
                }
            }
        } catch (error) {
            console.error('Error parsing job card:', error);
        }
    });
    
    console.log(`Found ${jobs.length} matching jobs`);
    
    // Send jobs to background script for processing
    if (jobs.length > 0) {
        chrome.runtime.sendMessage({
            action: 'jobsFound',
            jobs: jobs
        });
        
        // Auto-apply to first matching job
        if (preferences.active) {
            setTimeout(() => {
                applyToFirstEligibleJob(jobs);
            }, 3000 + Math.random() * 2000); // Random delay 3-5 seconds
        }
    }
}

// Check if job matches preferences
function matchesPreferences(job, preferences) {
    // Check job title
    const titleMatch = preferences.jobTitles.some(title => 
        job.title.toLowerCase().includes(title.toLowerCase())
    );
    
    if (!titleMatch && preferences.jobTitles.length > 0 && preferences.jobTitles[0] !== '') {
        return false;
    }
    
    // Check location
    if (preferences.locations && preferences.locations.length > 0 && preferences.locations[0] !== '') {
        const locationMatch = preferences.locations.some(location => 
            job.location.toLowerCase().includes(location.toLowerCase()) ||
            location.toLowerCase() === 'remote' && job.location.toLowerCase().includes('remote')
        );
        
        if (!locationMatch) {
            return false;
        }
    }
    
    return true;
}

// Apply to first eligible job
async function applyToFirstEligibleJob(jobs) {
    for (const job of jobs) {
        // Check if already applied
        const alreadyApplied = await checkIfAlreadyApplied(job.id);
        if (!alreadyApplied) {
            // Click on the job to open details
            const jobCard = document.querySelector(`[data-job-id="${job.id}"]`) || 
                           document.getElementById(job.id);
            
            if (jobCard) {
                jobCard.click();
                
                // Wait for job details to load
                setTimeout(() => {
                    // Get full job description
                    const descriptionElement = document.querySelector('.jobs-description, .job-view-layout');
                    if (descriptionElement) {
                        job.description = descriptionElement.textContent;
                        
                        // Send to background for scoring and application
                        chrome.runtime.sendMessage({
                            action: 'processJobApplication',
                            job: job
                        });
                    }
                }, 2000);
                
                break; // Only apply to one job at a time
            }
        }
    }
}

// Check if already applied to this job
async function checkIfAlreadyApplied(jobId) {
    // Check for "Applied" badge
    const appliedBadge = document.querySelector(`[data-job-id="${jobId}"] .job-card-container__footer-item--applied`);
    if (appliedBadge) return true;
    
    // Check in storage
    const data = await chrome.storage.local.get(['appliedJobs']);
    const appliedJobs = data.appliedJobs || [];
    return appliedJobs.includes(jobId);
}

// Apply to specific job
async function applyToSpecificJob(jobId, jobData) {
    try {
        console.log('Applying to job:', jobData.title, 'at', jobData.company);
        
        // Find and click Easy Apply button
        const easyApplyButton = document.querySelector('[aria-label*="Easy Apply"], .jobs-apply-button');
        
        if (!easyApplyButton) {
            return { success: false, reason: 'Easy Apply button not found' };
        }
        
        easyApplyButton.click();
        
        // Wait for modal to open
        await waitForElement('.jobs-easy-apply-modal, .jobs-apply-form');
        
        // Fill application form
        const filled = await fillApplicationForm();
        
        if (!filled) {
            return { success: false, reason: 'Failed to fill form' };
        }
        
        // Submit application
        const submitted = await submitApplication();
        
        if (submitted) {
            // Mark as applied
            await markAsApplied(jobId);
            return { success: true };
        }
        
        return { success: false, reason: 'Failed to submit' };
        
    } catch (error) {
        console.error('Error applying to job:', error);
        return { success: false, reason: error.message };
    }
}

// Fill application form
async function fillApplicationForm() {
    try {
        // Wait for form to load
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Phone number field
        const phoneInput = document.querySelector('input[name*="phone"], input[type="tel"]');
        if (phoneInput && !phoneInput.value) {
            const userData = await chrome.storage.local.get(['userPhone']);
            if (userData.userPhone) {
                phoneInput.value = userData.userPhone;
                phoneInput.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }
        
        // Email field (usually pre-filled)
        const emailInput = document.querySelector('input[type="email"]');
        if (emailInput && !emailInput.value) {
            const userData = await chrome.storage.local.get(['userEmail']);
            if (userData.userEmail) {
                emailInput.value = userData.userEmail;
                emailInput.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }
        
        // Handle multi-step forms
        const nextButton = document.querySelector('[aria-label="Continue to next step"], [data-control-name="continue_unify"]');
        if (nextButton) {
            nextButton.click();
            await new Promise(resolve => setTimeout(resolve, 1500));
            
            // Check for additional steps
            return await fillApplicationForm(); // Recursive call for multi-step
        }
        
        return true;
    } catch (error) {
        console.error('Error filling form:', error);
        return false;
    }
}

// Submit application
async function submitApplication() {
    try {
        // Find submit button
        const submitButton = document.querySelector(
            '[aria-label="Submit application"], ' +
            '[data-control-name="submit_unify"], ' +
            'button[type="submit"]:not([aria-label*="Continue"])'
        );
        
        if (!submitButton) {
            console.log('Submit button not found');
            return false;
        }
        
        // Check if button is enabled
        if (submitButton.disabled) {
            console.log('Submit button is disabled');
            return false;
        }
        
        // Click submit
        submitButton.click();
        
        // Wait for confirmation
        await new Promise(resolve => setTimeout(resolve, 2000));
        
        // Check for success message
        const successMessage = document.querySelector(
            '.artdeco-modal__header h2[id*="success"], ' +
            '.jobs-easy-apply-modal__header h2:contains("Application sent")'
        );
        
        if (successMessage) {
            console.log('Application submitted successfully!');
            
            // Close modal
            const closeButton = document.querySelector('[aria-label="Dismiss"]');
            if (closeButton) closeButton.click();
            
            return true;
        }
        
        // Check if modal closed (also indicates success sometimes)
        const modalStillOpen = document.querySelector('.jobs-easy-apply-modal');
        if (!modalStillOpen) {
            return true;
        }
        
        return false;
    } catch (error) {
        console.error('Error submitting application:', error);
        return false;
    }
}

// Mark job as applied
async function markAsApplied(jobId) {
    const data = await chrome.storage.local.get(['appliedJobs']);
    const appliedJobs = data.appliedJobs || [];
    
    if (!appliedJobs.includes(jobId)) {
        appliedJobs.push(jobId);
        await chrome.storage.local.set({ appliedJobs: appliedJobs });
    }
}

// Helper function to wait for element
function waitForElement(selector, timeout = 5000) {
    return new Promise((resolve, reject) => {
        const startTime = Date.now();
        
        const checkInterval = setInterval(() => {
            const element = document.querySelector(selector);
            
            if (element) {
                clearInterval(checkInterval);
                resolve(element);
            } else if (Date.now() - startTime > timeout) {
                clearInterval(checkInterval);
                reject(new Error(`Element ${selector} not found after ${timeout}ms`));
            }
        }, 100);
    });
}

// Generate unique job ID if not available
function generateJobId() {
    return 'job_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
}

// Initialize
console.log('LinkedIn Auto-Apply content script loaded');