// Popup script for extension interface

let currentResumeText = '';
let currentJobDescription = '';

// Initialize popup
document.addEventListener('DOMContentLoaded', async () => {
    await checkAuthentication();
    await updateUsageStats();
    setupEventListeners();
    loadSettings();
});

// Check authentication status
async function checkAuthentication() {
    const user = await chrome.runtime.sendMessage({ action: 'getCurrentUser' });
    
    if (user) {
        // User is signed in
        document.getElementById('googleSignInBtn').classList.add('hidden');
        document.getElementById('userInfo').classList.remove('hidden');
        document.getElementById('userAvatar').src = user.picture || '';
        document.getElementById('userName').textContent = user.name || user.email;
    } else {
        // User is not signed in
        document.getElementById('googleSignInBtn').classList.remove('hidden');
        document.getElementById('userInfo').classList.add('hidden');
    }
}

// Setup all event listeners
function setupEventListeners() {
    // Google Sign In
    document.getElementById('googleSignInBtn')?.addEventListener('click', handleGoogleSignIn);
    
    // Mode switcher
    document.querySelectorAll('.mode-btn').forEach(btn => {
        btn.addEventListener('click', handleModeSwitch);
    });
    
    // Source buttons
    document.querySelectorAll('.source-btn').forEach(btn => {
        btn.addEventListener('click', handleSourceChange);
    });
    
    // Scan button
    document.getElementById('scanButton').addEventListener('click', handleScan);
    
    // Job Seeker controls
    document.getElementById('startAutoApply')?.addEventListener('click', startAutoApply);
    document.getElementById('stopAutoApply')?.addEventListener('click', stopAutoApply);
    document.getElementById('minScore')?.addEventListener('input', updateMinScore);
    document.getElementById('viewDashboard')?.addEventListener('click', openDashboard);
    
    // Settings
    document.getElementById('settingsIcon')?.addEventListener('click', openSettings);
    document.getElementById('closeSettings')?.addEventListener('click', closeSettings);
    
    // Results actions
    document.getElementById('exportButton')?.addEventListener('click', exportReport);
    document.getElementById('newScanButton')?.addEventListener('click', resetScanner);
    
    // Upgrade link
    document.getElementById('upgradeLink')?.addEventListener('click', handleUpgrade);
    document.getElementById('manageSub')?.addEventListener('click', handleUpgrade);
    
    // Save API key
    document.getElementById('apiKey')?.addEventListener('blur', saveApiKey);
}

// Handle mode switching
function handleModeSwitch(e) {
    const mode = e.currentTarget.dataset.mode;
    
    // Update button states
    document.querySelectorAll('.mode-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    e.currentTarget.classList.add('active');
    
    // Update UI based on mode
    if (mode === 'recruiter') {
        // Recruiter mode - for screening candidates
        document.querySelector('.scanner-section').classList.remove('hidden');
        document.querySelector('#jobDescription').placeholder = 'Paste the job requirements here...';
        document.querySelector('#scanButton').innerHTML = '<span class="scan-icon">🔍</span> Screen Candidate';
        
        // Show recruiter-specific sections
        const jobseekerSection = document.querySelector('#jobseekerSection');
        if (jobseekerSection) jobseekerSection.classList.add('hidden');
        
        // Update labels
        const jdLabel = document.querySelector('#jdLabel');
        if (jdLabel) jdLabel.textContent = 'Job Requirements';
        
    } else {
        // Job seeker mode - for checking ATS score
        document.querySelector('.scanner-section').classList.remove('hidden');
        document.querySelector('#jobDescription').placeholder = 'Paste the job description you\'re applying to...';
        document.querySelector('#scanButton').innerHTML = '<span class="scan-icon">🎯</span> Check My ATS Score';
        
        // Update labels for job seeker
        const jdLabel = document.querySelector('#jdLabel');
        if (jdLabel) jdLabel.textContent = 'Target Job Description';
        
        // Show job seeker specific features
        const jobseekerSection = document.querySelector('#jobseekerSection');
        if (jobseekerSection) {
            jobseekerSection.classList.remove('hidden');
            jobseekerSection.classList.add('visible');
        }
    }
    
    // Save preference
    chrome.storage.local.set({ preferredMode: mode });
}

// Handle Google Sign In
async function handleGoogleSignIn() {
    const btn = document.getElementById('googleSignInBtn');
    btn.disabled = true;
    btn.textContent = 'Signing in...';
    
    try {
        const result = await chrome.runtime.sendMessage({ action: 'authenticateWithGoogle' });
        
        if (result.success) {
            await checkAuthentication();
            await updateUsageStats();
        } else {
            alert('Sign in failed. Please try again.');
        }
    } catch (error) {
        console.error('Sign in error:', error);
        alert('Sign in failed. Please try again.');
    } finally {
        btn.disabled = false;
        btn.innerHTML = '<img src="https://www.google.com/favicon.ico" alt="Google" width="16" height="16"> Sign in with Google';
    }
}

// Handle source change
function handleSourceChange(e) {
    document.querySelectorAll('.source-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    e.target.classList.add('active');
    
    const source = e.target.dataset.source;
    const inputDiv = document.getElementById('resumeInput');
    
    switch(source) {
        case 'current':
            inputDiv.innerHTML = `
                <div class="current-page-info">
                    <p>📄 Ready to scan resume on current page</p>
                    <small>Works on LinkedIn, Naukri, PDF viewers</small>
                </div>`;
            break;
        case 'upload':
            inputDiv.innerHTML = `
                <div class="upload-area">
                    <p>📎 Click to upload PDF or drag & drop</p>
                    <input type="file" id="fileInput" accept=".pdf,.doc,.docx" style="display:none">
                </div>`;
            setupFileUpload();
            break;
        case 'paste':
            inputDiv.innerHTML = `
                <textarea class="paste-area" id="pasteArea" placeholder="Paste resume text here..."></textarea>`;
            break;
    }
}

// Setup file upload
function setupFileUpload() {
    const uploadArea = document.querySelector('.upload-area');
    const fileInput = document.getElementById('fileInput');
    
    uploadArea.addEventListener('click', () => fileInput.click());
    
    uploadArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadArea.classList.add('dragover');
    });
    
    uploadArea.addEventListener('dragleave', () => {
        uploadArea.classList.remove('dragover');
    });
    
    uploadArea.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadArea.classList.remove('dragover');
        handleFileUpload(e.dataTransfer.files[0]);
    });
    
    fileInput.addEventListener('change', (e) => {
        handleFileUpload(e.target.files[0]);
    });
}

// Handle file upload
function handleFileUpload(file) {
    if (!file) return;
    
    // For demo purposes, show file name
    const uploadArea = document.querySelector('.upload-area');
    uploadArea.innerHTML = `<p>✅ ${file.name}</p><small>Ready to scan</small>`;
    
    // In production, you'd parse the PDF here
    currentResumeText = `[Resume from ${file.name}]`;
}

// Handle scan
async function handleScan() {
    const jobDescription = document.getElementById('jobDescription').value.trim();
    
    if (!jobDescription) {
        showError('Please enter a job description');
        return;
    }
    
    // Check if user can scan
    const canScan = await chrome.runtime.sendMessage({ action: 'checkCanScan' });
    
    if (!canScan.allowed) {
        showError('Daily limit reached. Upgrade to Pro for unlimited scans.');
        document.getElementById('upgradeLink').click();
        return;
    }
    
    // Get resume text based on source
    const activeSource = document.querySelector('.source-btn.active').dataset.source;
    let resumeText = '';
    
    if (activeSource === 'current') {
        // Extract from current page
        resumeText = await extractFromCurrentPage();
    } else if (activeSource === 'paste') {
        resumeText = document.getElementById('pasteArea')?.value || '';
    } else if (activeSource === 'upload') {
        resumeText = currentResumeText;
    }
    
    if (!resumeText || resumeText.length < 50) {
        showError('Please provide a valid resume');
        return;
    }
    
    // Show scanning state
    const scanButton = document.getElementById('scanButton');
    scanButton.classList.add('scanning');
    scanButton.innerHTML = '<span class="scan-icon">⏳</span> Scanning...';
    scanButton.disabled = true;
    
    try {
        // Send to background for scoring
        const response = await chrome.runtime.sendMessage({
            action: 'scoreResume',
            resumeText: resumeText,
            jobDescription: jobDescription
        });
        
        if (response.success) {
            displayResults(response.result);
            await updateUsageStats();
        } else {
            showError(response.error || 'Failed to scan resume');
        }
    } catch (error) {
        showError('An error occurred. Please try again.');
        console.error(error);
    } finally {
        scanButton.classList.remove('scanning');
        scanButton.innerHTML = '<span class="scan-icon">🔍</span> Scan Resume';
        scanButton.disabled = false;
    }
}

// Extract resume from current page
async function extractFromCurrentPage() {
    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        const response = await chrome.tabs.sendMessage(tab.id, { action: 'extractResume' });
        
        if (response.success) {
            return response.resumeText;
        } else {
            showError('Could not extract resume from this page');
            return '';
        }
    } catch (error) {
        showError('Please refresh the page and try again');
        return '';
    }
}

// Display results
function displayResults(result) {
    // Hide scanner, show results
    document.querySelector('.scanner-section').classList.add('hidden');
    document.getElementById('results').classList.remove('hidden');
    
    // Overall score
    const scoreCircle = document.getElementById('scoreCircle');
    const scoreValue = Math.round(result.overall_score);
    scoreCircle.innerHTML = `
        <span class="score-value">${scoreValue}%</span>
        <span class="score-label">Match</span>`;
    
    // Color based on score
    if (scoreValue >= 80) {
        scoreCircle.style.background = 'linear-gradient(135deg, #27ae60 0%, #2ecc71 100%)';
    } else if (scoreValue >= 60) {
        scoreCircle.style.background = 'linear-gradient(135deg, #f39c12 0%, #f1c40f 100%)';
    } else {
        scoreCircle.style.background = 'linear-gradient(135deg, #e74c3c 0%, #c0392b 100%)';
    }
    
    // Score breakdown
    updateProgressBar('skills', result.skills_score);
    updateProgressBar('experience', result.experience_score);
    updateProgressBar('education', result.education_score);
    
    // Insights
    const insightsList = document.getElementById('insightsList');
    insightsList.innerHTML = '';
    
    result.insights.forEach(insight => {
        const item = document.createElement('div');
        item.className = `insight-item ${insight.type}`;
        
        let icon = '•';
        if (insight.type === 'positive') icon = '✓';
        if (insight.type === 'negative') icon = '✗';
        if (insight.type === 'neutral') icon = '→';
        
        item.innerHTML = `
            <span class="insight-icon">${icon}</span>
            <span>${insight.text}</span>`;
        insightsList.appendChild(item);
    });
    
    // Add missing keywords if any
    if (result.missing_keywords && result.missing_keywords.length > 0) {
        const keywordsItem = document.createElement('div');
        keywordsItem.className = 'insight-item negative';
        keywordsItem.innerHTML = `
            <span class="insight-icon">!</span>
            <span>Missing keywords: ${result.missing_keywords.join(', ')}</span>`;
        insightsList.appendChild(keywordsItem);
    }
    
    // Store results for export
    currentJobDescription = document.getElementById('jobDescription').value;
    window.lastScanResult = result;
}

// Update progress bar
function updateProgressBar(type, score) {
    const progress = document.getElementById(`${type}Progress`);
    const scoreText = document.getElementById(`${type}Score`);
    
    progress.style.width = `${score}%`;
    scoreText.textContent = `${Math.round(score)}%`;
    
    // Color based on score
    if (score >= 80) {
        progress.style.background = 'linear-gradient(90deg, #27ae60 0%, #2ecc71 100%)';
    } else if (score >= 60) {
        progress.style.background = 'linear-gradient(90deg, #f39c12 0%, #f1c40f 100%)';
    } else {
        progress.style.background = 'linear-gradient(90deg, #e74c3c 0%, #c0392b 100%)';
    }
}

// Reset scanner
function resetScanner() {
    document.querySelector('.scanner-section').classList.remove('hidden');
    document.getElementById('results').classList.add('hidden');
    document.getElementById('jobDescription').value = '';
    document.querySelector('.source-btn[data-source="current"]').click();
}

// Export report
function exportReport() {
    if (!window.lastScanResult) return;
    
    const report = `
RESUME SCAN REPORT
==================
Generated: ${new Date().toLocaleString()}

JOB DESCRIPTION:
${currentJobDescription}

SCAN RESULTS:
Overall Match: ${window.lastScanResult.overall_score}%
Skills Match: ${window.lastScanResult.skills_score}%
Experience Match: ${window.lastScanResult.experience_score}%
Education Match: ${window.lastScanResult.education_score}%

KEY INSIGHTS:
${window.lastScanResult.insights.map(i => `- [${i.type.toUpperCase()}] ${i.text}`).join('\n')}

MISSING KEYWORDS:
${window.lastScanResult.missing_keywords.join(', ')}

RECOMMENDATION:
${window.lastScanResult.recommendation}
`;
    
    // Download as text file
    const blob = new Blob([report], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `resume-scan-${Date.now()}.txt`;
    a.click();
}

// Update usage stats
async function updateUsageStats() {
    const stats = await chrome.runtime.sendMessage({ action: 'getUsageStats' });
    
    document.getElementById('scansUsed').textContent = stats.scansUsed;
    
    if (stats.isPro) {
        document.getElementById('planBadge').textContent = 'PRO';
        document.getElementById('planBadge').classList.add('pro');
        document.getElementById('usageLimit').innerHTML = '<p>Unlimited Scans ✨</p>';
        document.getElementById('currentPlan').textContent = 'Pro';
    }
}

// Settings functions
function openSettings() {
    document.getElementById('settingsModal').classList.remove('hidden');
}

function closeSettings() {
    document.getElementById('settingsModal').classList.add('hidden');
}

async function loadSettings() {
    const data = await chrome.storage.local.get(['apiKey', 'autoExtract', 'detailedMode']);
    
    if (data.apiKey) {
        document.getElementById('apiKey').value = data.apiKey;
    }
    
    document.getElementById('autoExtract').checked = data.autoExtract !== false;
    document.getElementById('detailedMode').checked = data.detailedMode === true;
}

async function saveApiKey() {
    const apiKey = document.getElementById('apiKey').value.trim();
    
    if (apiKey) {
        await chrome.runtime.sendMessage({ action: 'saveApiKey', apiKey: apiKey });
        showSuccess('API Key saved');
    }
}

// Handle upgrade
async function handleUpgrade(e) {
    e.preventDefault();
    await chrome.runtime.sendMessage({ action: 'upgradeToPro' });
}

// Show error message
function showError(message) {
    const toast = document.createElement('div');
    toast.className = 'toast error';
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #e74c3c;
        color: white;
        padding: 12px 20px;
        border-radius: 6px;
        z-index: 10000;
        animation: slideIn 0.3s;
    `;
    document.body.appendChild(toast);
    
    setTimeout(() => toast.remove(), 3000);
}

// Show success message
function showSuccess(message) {
    const toast = document.createElement('div');
    toast.className = 'toast success';
    toast.textContent = message;
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #27ae60;
        color: white;
        padding: 12px 20px;
        border-radius: 6px;
        z-index: 10000;
        animation: slideIn 0.3s;
    `;
    document.body.appendChild(toast);
    
    setTimeout(() => toast.remove(), 3000);
}

// Job Seeker Functions
function updateMinScore(e) {
    document.getElementById('minScoreValue').textContent = e.target.value + '%';
}

async function startAutoApply() {
    // Get preferences
    const jobTitles = document.getElementById('jobTitles').value.split(',').map(t => t.trim());
    const locations = document.getElementById('locations').value.split(',').map(l => l.trim());
    const experienceLevel = document.getElementById('experienceLevel').value;
    const minScore = parseInt(document.getElementById('minScore').value);
    
    if (jobTitles.length === 0 || jobTitles[0] === '') {
        alert('Please enter at least one job title');
        return;
    }
    
    // Save preferences
    const preferences = {
        jobTitles,
        locations,
        experienceLevel,
        minScore,
        active: true
    };
    
    await chrome.storage.local.set({ autoApplyPreferences: preferences });
    
    // Update UI
    document.getElementById('startAutoApply').classList.add('hidden');
    document.getElementById('stopAutoApply').classList.remove('hidden');
    document.getElementById('applyStatus').classList.remove('hidden');
    
    // Send message to background script to start auto-apply
    chrome.runtime.sendMessage({ 
        action: 'startAutoApply',
        preferences 
    });
    
    showSuccess('Auto-Apply started! We\'ll find and apply to matching jobs.');
}

async function stopAutoApply() {
    // Update preferences
    const data = await chrome.storage.local.get(['autoApplyPreferences']);
    if (data.autoApplyPreferences) {
        data.autoApplyPreferences.active = false;
        await chrome.storage.local.set({ autoApplyPreferences: data.autoApplyPreferences });
    }
    
    // Update UI
    document.getElementById('startAutoApply').classList.remove('hidden');
    document.getElementById('stopAutoApply').classList.add('hidden');
    
    // Send message to background script
    chrome.runtime.sendMessage({ action: 'stopAutoApply' });
    
    showSuccess('Auto-Apply stopped');
}

function openDashboard() {
    chrome.tabs.create({ url: chrome.runtime.getURL('dashboard.html') });
}

// Listen for updates from background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'updateAutoApplyStats') {
        document.getElementById('jobsFound').textContent = request.stats.found;
        document.getElementById('jobsApplied').textContent = request.stats.applied;
        document.getElementById('jobsSkipped').textContent = request.stats.skipped;
        
        // Add to recent applications list
        if (request.recentApplication) {
            addRecentApplication(request.recentApplication);
        }
    }
});

function addRecentApplication(application) {
    const list = document.getElementById('applicationsList');
    const item = document.createElement('div');
    item.className = 'application-item';
    item.innerHTML = `
        <span class="time">${new Date().toLocaleTimeString()}</span>
        <div class="company">${application.company}</div>
        <div class="position">${application.position}</div>
    `;
    
    list.insertBefore(item, list.firstChild);
    
    // Keep only last 5
    while (list.children.length > 5) {
        list.removeChild(list.lastChild);
    }
}